#!/bin/sh

JSON="/www/networkmap.json"
OUT="/tmp/networkmap_new.json"

echo "Collecting DHCP leases..."
DHCP_JSON=$(awk '{
    mac=toupper($2);
    ip=$3;
    name=$4;
    printf "{\"id\":\"%s\",\"mac\":\"%s\",\"ip\":\"%s\"}\n", name, mac, ip
}' /tmp/dhcp.leases | jq -s 'map({(.id): {mac, ip}}) | add')

echo "Collecting IPv6 neighbors..."
IPV6_JSON=$(ip -6 neigh | awk '$5 != "" {
    mac=toupper($5);
    ipv6=toupper($1);
    printf "{\"mac\":\"%s\",\"ipv6\":\"%s\"}\n", mac, ipv6
}' | jq -s 'map({(.mac): .ipv6}) | add')

echo "Updating networkmap.json..."
if jq \
  --argjson dhcp "$DHCP_JSON" \
  --argjson ipv6 "$IPV6_JSON" \
  '
  .nodes |= map(
    . as $node |

    # Update MAC (if DHCP knows this device)
    (if $dhcp[$node.id] then $dhcp[$node.id].mac else $node.mac end) as $mac |

    # Update IPv4
    (if $dhcp[$node.id] then $dhcp[$node.id].ip else $node.ip end) as $ipv4 |

    # Update IPv6 (match by MAC)
    (if $mac != null and $ipv6[$mac] then $ipv6[$mac] else $node.ipv6 end) as $ipv6 |

    # Apply updates
    $node
    | .mac = $mac
    | .ip = $ipv4
    | .ipv6 = ($ipv6 // "none")
  )
  ' "$JSON" > "$OUT"
then
    mv "$OUT" "$JSON"
    echo "Done. networkmap.json updated."
else
    echo "ERROR: jq failed. Original JSON preserved."
    rm -f "$OUT"
    exit 1
fi
