#!/bin/ash

TOPOLOGY="/www/networkmap-topology.json"
OUTPUT="/www/networkmap.json"

awk -v topo="$TOPOLOGY" '
BEGIN {
    # initialize arrays
    state = "idle"
}

# 1. Read topology file line-by-line
FNR == NR {
    line = $0

    ########################################
    # 1. Detect entry into "nodes" array
    ########################################
    if (index(line, "\"nodes\"") > 0) {
        state = "reading_nodes"
        next
    }

    ########################################
    # 2. Detect entry into "links" array
    ########################################
    if (index(line, "\"links\"") > 0) {
    state = "reading_links"

    # If the '[' is on the same line, start collecting immediately
    if (index(line, "[") > 0) {
            linksCollect = 1
            linksRaw = ""
        }

        next
    }

    ########################################
    # 3. Start of a node block
    ########################################
    if (state == "reading_nodes") {
        if (index(line, "{") > 0) {
            state = "reading_node_block"
            block = ""
        }
    }

    ########################################
    # 4. Node-block accumulator + processor
    ########################################
    if (state == "reading_node_block") {
        block = block line "\n"

        if (index(line, "}") > 0) {
            # Extract fields from block
            id = ""; mac = ""; type = ""; conn = ""; ip = "";

            # Split block into lines
            n = split(block, arr, "\n");
            for (i = 1; i <= n; i++) {
                l = arr[i];
                gsub(/,$/, "", l);   # <-- strip trailing commas

                # id
                if (index(l, "\"id\"") > 0) {
                    gsub(/.*"id":[[:space:]]*"/, "", l);
                    gsub(/".*/, "", l);
                    id = l;
                }

                # mac
                if (index(l, "\"mac\"") > 0) {
                    gsub(/.*"mac":[[:space:]]*"/, "", l);
                    gsub(/".*/, "", l);
                    mac = toupper(l);
                }

                # type
                if (index(l, "\"type\"") > 0) {
                    gsub(/.*"type":[[:space:]]*"/, "", l);
                    gsub(/".*/, "", l);
                    type = l;
                }

                # connection
                if (index(l, "\"connection\"") > 0) {
                    gsub(/.*"connection":[[:space:]]*"/, "", l);
                    gsub(/".*/, "", l);
                    conn = l;
                }

                # ip
                if (index(l, "\"ip\"") > 0) {
                    gsub(/.*"ip":[[:space:]]*"/, "", l);
                    gsub(/".*/, "", l);
                    ip = l;
                }
            }

            # Store node identity
            if (mac != "") {
                mac = toupper(mac)
                nodeID[mac] = id;
                nodeType[mac] = type;
                nodeConn[mac] = conn;
                nodeIP[mac] = ip;
                nodeSeen[mac] = 0;
            }
            state = "reading_nodes"
        }

        next
    }

    ########################################
    # 5. Links block accumulator
    ########################################
    if (state == "reading_links") {

        # Start collecting if we see '['
        if (index(line, "[") > 0 && linksCollect != 1) {
            linksCollect = 1
            linksRaw = ""
            next
        }

        # Collect lines
        if (linksCollect == 1) {
            if (index(line, "]") > 0) {
                linksCollect = 0
                state = "idle"
            } else {
                linksRaw = linksRaw line "\n"
            }
        }

        next
    }
}   # <-- THIS closes FNR == NR

# 2. Read DHCP leases
FNR != NR {
    mac = toupper($2)
    ipv4 = $3
    host = $4

    leaseIPv4[mac] = ipv4
    leaseHost[mac] = host
    nodeSeen[mac] = systime()

    next
}

END {
    ########################################
    # 1. Gather IPv6 neighbors
    ########################################
    while (( "ip -6 neigh" | getline line ) > 0 ) {
        split(line, a, " ");
        ip6 = a[1];
        mac = toupper(a[5]);
        if (mac != "") ipv6list[mac] = ipv6list[mac] ip6 "\n";
    }
    close("ip -6 neigh");

    ########################################
    # 1b. Auto-detect router MAC + IPv6
    ########################################

    # Detect router MAC from br-lan
    router_mac = ""
    while (( "ip link show br-lan" | getline line ) > 0 ) {
        if (index(line, "link/ether") > 0) {
            split(line, a, " ")
            router_mac = toupper(a[2])
        }
    }
    close("ip link show br-lan")

    # Detect router IPv6 (link-local) from br-lan
    if (router_mac != "") {
        while (( "ip -6 addr show br-lan" | getline line ) > 0 ) {
            if (index(line, "fe80::") > 0) {
                split(line, a, " ")
                router_ipv6 = a[2]
                sub(/\/.*/, "", router_ipv6)  # strip /64
                ipv6list[router_mac] = ipv6list[router_mac] router_ipv6 "\n"
            }
        }
        close("ip -6 addr show br-lan")
    }

    ########################################
    # 2. Gather IPv4 neighbors (ARP)
    ########################################
    while (( "ip neigh" | getline line ) > 0 ) {
        split(line, a, " ");
        ip4 = a[1];
        mac = toupper(a[5]);
        if (mac != "") neigh4[mac] = neigh4[mac] ip4 "\n";
    }
    close("ip neigh");

    ########################################
    # 3. Add unknown devices (DHCP-only)
    ########################################
    for (m in leaseIPv4) {
        if (!(m in nodeID)) {
            name = leaseHost[m];
            if (name == "" || name == "*") name = "Unknown-" m;
            nodeID[m] = name;
            nodeType[m] = "unknown";
            nodeConn[m] = "unknown";
            nodeIP[m] = leaseIPv4[m];
            nodeSeen[m] = systime();
        }
    }

    ########################################
    # 4. Output JSON
    ########################################
    print "{";

    ########################################
    # LINKS
    ########################################
    print "\"links\":[";
    printf "%s", linksRaw;   # already newline-separated
    print "],";

    ########################################
    # NODES
    ########################################
    print "\"nodes\":[";

    first = 1;
    for (mac in nodeID) {
        if (!first) print ",";
        first = 0;

        printf "{";
        printf "\"id\":\"%s\",", nodeID[mac];
        printf "\"ip\":\"%s\",", nodeIP[mac];
        printf "\"type\":\"%s\",", nodeType[mac];
        printf "\"connection\":\"%s\",", nodeConn[mac];
        printf "\"online\":%s,", (nodeSeen[mac] > 0 ? "true" : "false");
        printf "\"lastSeen\":%d,", nodeSeen[mac];
        printf "\"mac\":\"%s\",", mac;

        ########################################
        # IPv6 array
        ########################################
        printf "\"ipv6\":[";
        if (ipv6list[mac] != "") {
            n = split(ipv6list[mac], arr, "\n");
            c = 0;
            for (i = 1; i <= n; i++) {
                if (arr[i] != "") {
                    if (c > 0) printf ",";
                    printf "\"%s\"", arr[i];
                    c++;
                }
            }
        }
        printf "]";

        printf "}";
    }

    print "]";
    print "}";
}
' "$TOPOLOGY" /tmp/dhcp.leases > "$OUTPUT"
