// ------------------------------------------------------------
// 1. Load JSON data
// ------------------------------------------------------------
let allNodes = [];
let allLinks = [];

let visibleNodes = [];
let visibleLinks = [];

d3.json("networkmap.json").then(data => {
    allNodes = data.nodes;
    allLinks = data.links;

    // ⭐ FIX: initialize missing positions BEFORE initGraph()
    // Find the router (center node)
    const router = allNodes.find(n => n.type === "router");

    // Choose a radius for the ring
    const RADIUS = 250;

    // Count how many offline nodes need placement
    const offlineNodes = allNodes.filter(n => !n.online);

    // Spread them evenly around the circle
    offlineNodes.forEach((n, i) => {
        const angle = (i / offlineNodes.length) * 2 * Math.PI;

        n.x = router.x + RADIUS * Math.cos(angle);
        n.y = router.y + RADIUS * Math.sin(angle);

        n.vx = 0;
        n.vy = 0;
    });


    initGraph();

    // Close button logic to close the info bubble
    document.getElementById("tooltipClose").onclick = () => {
        document.getElementById("nodeTooltip").style.display = "none";
    };

    svg.on("click", (event) => {
        if (event.target.tagName === "svg") {
            document.getElementById("nodeTooltip").style.display = "none";
        }
    });
});

// ------------------------------------------------------------
// 2. Set up SVG + groups
// ------------------------------------------------------------
const svg = d3.select("#lanmap");
const width = +svg.attr("width");
const height = +svg.attr("height");

const linkGroup = svg.append("g").attr("class", "links");
const nodeGroup = svg.append("g").attr("class", "nodes");

let nodeElements = null;
let linkElements = null;

// ------------------------------------------------------------
// 3. Force simulation
// ------------------------------------------------------------
const simulation = d3.forceSimulation()
    .force("link", d3.forceLink().id(d => d.id).distance(80))
    .force("charge", d3.forceManyBody().strength(-200))
    .force("center", d3.forceCenter(width / 2, height / 2));

// ------------------------------------------------------------
// 4. Filtering logic (YOUR CODE)
// ------------------------------------------------------------
function filterNodes(showOffline) {
    const THIRTY_DAYS = 30 * 24 * 60 * 60 * 1000;
    const now = Date.now();

    return allNodes.filter(node => {
        if (node.online) return true;

        const offlineTime = now - node.lastSeen;

        if (showOffline && offlineTime < THIRTY_DAYS) {
            node.dimmed = true;
            return true;
        }

        return false;
    });
}

function filterLinks(visibleNodes) {
    const visibleIds = new Set(visibleNodes.map(n => n.id));
    return allLinks.filter(link =>
        visibleIds.has(link.source.id || link.source) &&
        visibleIds.has(link.target.id || link.target)
    );
}

// ------------------------------------------------------------
// 5. Toggle handler (YOUR CODE)
// ------------------------------------------------------------
//document.getElementById("toggleOffline").addEventListener("change", e => {
//    const showOffline = e.target.checked;

//    visibleNodes = filterNodes(showOffline);
//    visibleLinks = filterLinks(visibleNodes);

//    updateGraph(visibleNodes, visibleLinks);
//});

// ------------------------------------------------------------
// 6. Graph initialization
// ------------------------------------------------------------
function initGraph() {
    visibleNodes = filterNodes(false); // default: online only
    visibleLinks = filterLinks(visibleNodes);

    updateGraph(visibleNodes, visibleLinks);

    simulation.on("tick", () => {
        linkElements
            .attr("x1", d => d.source.x)
            .attr("y1", d => d.source.y)
            .attr("x2", d => d.target.x)
            .attr("y2", d => d.target.y);

        nodeElements
            .attr("cx", d => d.x)
            .attr("cy", d => d.y);
    });
}

// ------------------------------------------------------------
// 7. updateGraph() (YOUR CODE)
// ------------------------------------------------------------
function updateGraph(nodes, links) {
    // JOIN
    nodeElements = nodeGroup.selectAll("circle")
        .data(nodes, d => d.id);

    linkElements = linkGroup.selectAll("line")
        .data(links);

    // EXIT
    nodeElements.exit().remove();
    linkElements.exit().remove();

    // ENTER
    const nodeEnter = nodeElements.enter()
        .append("circle")
        .attr("r", 8)
        .attr("fill", d => colorScale(d.type));

    // Hover tooltip
    nodeEnter.append("title")
        .text(d => `${d.id}\n${d.ip}\n${d.ipv6 || ""}`);

    // Click handler
    nodeEnter.on("click", (event, d) => {
        event.stopPropagation(); // ← prevents auto-close

        const tooltip = document.getElementById("nodeTooltip");
        const content = document.getElementById("tooltipContent");

    // Fill content
    content.innerHTML = `
        <strong>${d.id}</strong><br>
        IPv4: ${d.ip}<br>
        IPv6: ${d.ipv6 || "none"}<br>
        MAC: ${d.mac || "unknown"}<br>
        Connection: ${d.connection}<br>
        Type: ${d.type}
    `;

    // Position near the cursor
    tooltip.style.left = (event.pageX + 12) + "px";
    tooltip.style.top = (event.pageY + 12) + "px";

    // Show it
    tooltip.style.display = "block";
    });


    // ENTER for links
    const linkEnter = linkElements.enter()
        .append("line")
        .attr("stroke-width", 1.5);

    // MERGE
    nodeElements = nodeEnter.merge(nodeElements);
    linkElements = linkEnter.merge(linkElements);

    // LINK STYLE: wifi = animated radio waves, ethernet = solid
    linkElements
      .attr("stroke", "#888888")
      .attr("stroke-width", 2)
      .attr("stroke-dasharray", d => {
          const tgtId = d.target.id || d.target;
          const tgt = nodes.find(n => n.id === tgtId);

          if (tgt?.connection === "wifi-b") return "8 6";   // slower 2.4 GHz wifi (802.11b, 802.11a, 802.11g)
          if (tgt?.connection === "wifi") return "4 4";     // normal 2.4 , 5 or 6 GHz wifi (802.11n, 802.11ac, 801.11ax, and beyond)
          return "0";                                       // ethernet
      })
      .attr("class", d => {
          const tgtId = d.target.id || d.target;
          const tgt = nodes.find(n => n.id === tgtId);

          if (tgt?.connection === "wifi-b") return "wifi-b-link";
          if (tgt?.connection === "wifi") return "wifi-link";
          return "";
    });

    // NODE STYLE
    nodeElements.attr("opacity", d => d.online ? 1 : 0.25);
    linkElements.attr("opacity", d => (d.source.online && d.target.online) ? 1 : 0.2);

    // RESTART SIMULATION
    simulation.nodes(nodes);
    simulation.force("link").links(links);
    simulation.alpha(1).restart();
}

// ------------------------------------------------------------
// 8. Color scale (simple example)
// ------------------------------------------------------------
function colorScale(type) {
    const colors = {
        router: "#ff9800",
        console: "#4caf50",
        phone: "#2196f3",
        pc: "#9c27b0",
        tablet: "#3f51b5",
        printer: "#795548",
        camera: "#e91e63",
        iot: "#009688",
        vm: "#607d8b",
        tv: "#8bc34a",
        streaming: "#00bcd4",
        switch: "#cddc39",
        modem: "#f44336"
    };
    return colors[type] || "#ffffff";
}
