#!/bin/bash

CONFIG="/home/user/scripts/dynamicdns.conf"
IP_FILE="/home/user/scripts/.last_ip"
DATE=$(date '+%Y-%m-%d %H:%M:%S')

update_ddns() {
    local URL=$1
    local ATTEMPTS=0
    local MAX_ATTEMPTS=10
    local SUCCESS=0

    while [ $ATTEMPTS -lt $MAX_ATTEMPTS ]; do
        RESPONSE=$(curl -s "$URL")
        if [[ "$RESPONSE" == *"nochg"* || "$RESPONSE" == *"good"* ]]; then
            echo "$RESPONSE"
            SUCCESS=1
            break
        fi
        ATTEMPTS=$((ATTEMPTS + 1))
        sleep 3
    done

    if [ $SUCCESS -eq 0 ]; then
        echo "[ERROR] Can't reach Hurricane Electric DNS (dns.he.net) after $MAX_ATTEMPTS attempts."
        echo "[ERROR] Please check network connectivity and try again."
    fi
}


echo "[$DATE] Starting DDNS check..."

# Get current IPs
CURRENT_IPV4=$(curl -s https://api.ipify.org)
CURRENT_IPV6=$(curl -s https://api6.ipify.org)

echo "[$DATE] Current IPv4: $CURRENT_IPV4"
echo "[$DATE] Current IPv6: $CURRENT_IPV6"

# Load previous IPs
if [ -f "$IP_FILE" ]; then
    read -r LAST_IPV4 LAST_IPV6 < "$IP_FILE"
else
    LAST_IPV4=""
    LAST_IPV6=""
fi

# Only update if IPs changed
if [ "$CURRENT_IPV4" != "$LAST_IPV4" ] || [ "$CURRENT_IPV6" != "$LAST_IPV6" ]; then
    echo "[$DATE] IP change detected. Updating subdomains..."

    while read -r DOMAIN PASSWORD; do
        # Skip comments and reserved entries
        [[ "$DOMAIN" =~ ^#.*$ || "$PASSWORD" == "reserved" ]] && continue

        echo "[$DATE] → Updating IPv4 for $DOMAIN..."
        update_ddns "https://dyn.dns.he.net/nic/update?hostname=$DOMAIN&password=$PASSWORD&myip=$CURRENT_IPV4"
        sleep 5   # pause for 5 seconds before the next request

        echo "[$DATE] → Updating IPv6 for $DOMAIN..."
        update_ddns "https://dyn.dns.he.net/nic/update?hostname=$DOMAIN&password=$PASSWORD&myip=$CURRENT_IPV6"
        sleep 5   # pause for 5 seconds before the next request

        echo "[$DATE] ✓ Finished updating $DOMAIN"
    done < "$CONFIG"

    echo "$CURRENT_IPV4 $CURRENT_IPV6" > "$IP_FILE"
else
    echo "[$DATE] No IP change detected. Skipping update."
fi

echo "[$DATE] DDNS check complete."
